/***************************************************************************
 *
 *   BSD LICENSE
 *
 *   Copyright(c) 2023 Intel Corporation. All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Intel Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/

#ifndef QATZIP_KM_H_
#define QATZIP_KM_H_

#include "qatzip.h"

typedef int (*QzInitFn)(QzSession_T *sess, unsigned char sw_backup);

typedef int (*QzSetupSessionFn)(QzSession_T *sess, QzSessionParams_T *params);

typedef int (*QzCompressFn)(QzSession_T *sess,
                            const unsigned char *src,
                            unsigned int *src_len,
                            unsigned char *dest,
                            unsigned int *dest_len,
                            unsigned int last);

typedef int (*QzCompressCrcFn)(QzSession_T *sess,
                               const unsigned char *src,
                               unsigned int *src_len,
                               unsigned char *dest,
                               unsigned int *dest_len,
                               unsigned int last,
                               unsigned long *crc);

typedef int (*QzDecompressFn)(QzSession_T *sess,
                              const unsigned char *src,
                              unsigned int *src_len,
                              unsigned char *dest,
                              unsigned int *dest_len);

typedef int (*QzTeardownSessionFn)(QzSession_T *sess);

typedef int (*QzCloseFn)(QzSession_T *sess);

typedef int (*QzGetStatusFn)(QzSession_T *sess, QzStatus_T *status);

typedef unsigned int (*QzMaxCompressedLengthFn)(unsigned int src_sz,
                                                QzSession_T *sess);

typedef int (*QzSetDefaultsFn)(QzSessionParams_T *defaults);

typedef int (*QzGetDefaultsFn)(QzSessionParams_T *defaults);

typedef void *(*QzMallocFn)(size_t sz, int numa, int forced_pinned);

typedef void (*QzFreeFn)(void *m);

typedef int (*QzMemFindAddrFn)(unsigned char *a);

typedef int (*QzCompressStreamFn)(QzSession_T *sess,
                                  QzStream_T *strm,
                                  unsigned int last);

typedef int (*QzDecompressStreamFn)(QzSession_T *sess,
                                    QzStream_T *strm,
                                    unsigned int last);

typedef int (*QzEndStreamFn)(QzSession_T *sess, QzStream_T *strm);

typedef int (*QzGetSoftwareComponentVersionListFn)(
    QzSoftwareVersionInfo_T *api_info,
    unsigned int *num_elem);

typedef int (*QzGetSoftwareComponentCountFn)(unsigned int *num_components);

typedef int (*QzSetupSessionDeflateFn)(QzSession_T *sess,
                                       QzSessionParamsDeflate_T *params);

typedef int (*QzSetupSessionLZ4Fn)(QzSession_T *sess,
                                   QzSessionParamsLZ4_T *params);

typedef int (*QzSetupSessionLZ4SFn)(QzSession_T *sess,
                                    QzSessionParamsLZ4S_T *params);

typedef int (*QzSetDefaultsDeflateFn)(QzSessionParamsDeflate_T *defaults);

typedef int (*QzSetDefaultsLZ4Fn)(QzSessionParamsLZ4_T *defaults);

typedef int (*QzSetDefaultsLZ4SFn)(QzSessionParamsLZ4S_T *defaults);

typedef int (*QzGetDefaultsDeflateFn)(QzSessionParamsDeflate_T *defaults);

typedef int (*QzGetDefaultsLZ4Fn)(QzSessionParamsLZ4_T *defaults);

typedef int (*QzGetDefaultsLZ4SFn)(QzSessionParamsLZ4S_T *defaults);

typedef int (*QzCompressExtFn)(QzSession_T *sess,
                               const unsigned char *src,
                               unsigned int *src_len,
                               unsigned char *dest,
                               unsigned int *dest_len,
                               unsigned int last,
                               uint64_t *ext_rc);

typedef int (*QzDecompressExtFn)(QzSession_T *sess,
                                 const unsigned char *src,
                                 unsigned int *src_len,
                                 unsigned char *dest,
                                 unsigned int *dest_len,
                                 uint64_t *ext_rc);

typedef int (*QzCompressCrc64Fn)(QzSession_T *sess,
                                 const unsigned char *src,
                                 unsigned int *src_len,
                                 unsigned char *dest,
                                 unsigned int *dest_len,
                                 unsigned int last,
                                 uint64_t *crc);

typedef int (*QzCompressCrc64ExtFn)(QzSession_T *sess,
                                    const unsigned char *src,
                                    unsigned int *src_len,
                                    unsigned char *dest,
                                    unsigned int *dest_len,
                                    unsigned int last,
                                    uint64_t *crc,
                                    uint64_t *ext_rc);

typedef int (*QzDecompressCrcExtFn)(QzSession_T *sess,
                                    const unsigned char *src,
                                    unsigned int *src_len,
                                    unsigned char *dest,
                                    unsigned int *dest_len,
                                    unsigned long *crc,
                                    uint64_t *ext_rc);

typedef int (*QzDecompressCrcFn)(QzSession_T *sess,
                                 const unsigned char *src,
                                 unsigned int *src_len,
                                 unsigned char *dest,
                                 unsigned int *dest_len,
                                 unsigned long *crc);

typedef int (*QzDecompressCrc64ExtFn)(QzSession_T *sess,
                                      const unsigned char *src,
                                      unsigned int *src_len,
                                      unsigned char *dest,
                                      unsigned int *dest_len,
                                      uint64_t *crc,
                                      uint64_t *ext_rc);

typedef int (*QzDecompressCrc64Fn)(QzSession_T *sess,
                                   const unsigned char *src,
                                   unsigned int *src_len,
                                   unsigned char *dest,
                                   unsigned int *dest_len,
                                   uint64_t *crc);

typedef int (*QzGetSessionCrc64Config)(QzSession_T *sess,
                                       QzCrc64Config_T *crc64_config);

typedef int (*QzSetSessionCrc64Config)(QzSession_T *sess,
                                       QzCrc64Config_T *crc64_config);

typedef int (*QzAllocateMetadataFn)(QzMetadataBlob_T *metadata,
                                    size_t data_size,
                                    uint32_t hw_buff_sz);

typedef int (*QzFreeMetadataFn)(QzMetadataBlob_T metadata);

typedef int (*QzMetadataBlockReadFn)(uint32_t block_num,
                                     QzMetadataBlob_T metadata,
                                     uint32_t *block_offset,
                                     uint32_t *block_size,
                                     uint32_t *block_flags,
                                     uint32_t *block_hash);

typedef int (*QzMetadataBlockWriteFn)(uint32_t block_num,
                                      QzMetadataBlob_T metadata,
                                      uint32_t *block_offset,
                                      uint32_t *block_size,
                                      uint32_t *block_flags,
                                      uint32_t *block_hash);

typedef int (*QzCompressWithMetadataExtFn)(QzSession_T *sess,
                                           const unsigned char *src,
                                           unsigned int *src_len,
                                           unsigned char *dest,
                                           unsigned int *dest_len,
                                           unsigned int last,
                                           uint64_t *ext_rc,
                                           QzMetadataBlob_T metadata,
                                           uint32_t hw_buff_sz_override,
                                           uint32_t comp_thrshold);

typedef int (*QzDecompressWithMetadataExtFn)(QzSession_T *sess,
                                             const unsigned char *src,
                                             unsigned int *src_len,
                                             unsigned char *dest,
                                             unsigned int *dest_len,
                                             uint64_t *ext_rc,
                                             QzMetadataBlob_T metadata,
                                             uint32_t hw_buff_sz_override);

typedef int (*QzGetSessionCrc32Config)(QzSession_T *sess,
                                       QzCrc32Config_T *crc32_config);

typedef int (*QzSetSessionCrc32Config)(QzSession_T *sess,
                                       QzCrc32Config_T *crc32_config);

typedef int (*QzMetadataBlockGetCrc64Fn)(uint32_t block_num,
                                         QzMetadataBlob_T metadata,
                                         uint64_t *input_crc,
                                         uint64_t *output_crc);

typedef int (*QzMetadataBlockGetCrc32Fn)(uint32_t block_num,
                                         QzMetadataBlob_T metadata,
                                         uint32_t *input_crc,
                                         uint32_t *output_crc);

typedef struct _QZ_FUNCTION_TABLE
{
    const QzInitFn qzInit;
    const QzSetupSessionFn qzSetupSession;
    const QzCompressFn qzCompress;
    const QzCompressCrcFn qzCompressCrc;
    const QzDecompressFn qzDecompress;
    const QzTeardownSessionFn qzTeardownSession;
    const QzCloseFn qzClose;
    const QzGetStatusFn qzGetStatus;
    const QzMaxCompressedLengthFn qzMaxCompressedLength;
    const QzSetDefaultsFn qzSetDefaults;
    const QzGetDefaultsFn qzGetDefaults;
    const QzMallocFn qzMalloc;
    const QzFreeFn qzFree;
    const QzMemFindAddrFn qzMemFindAddr;
    const QzCompressStreamFn qzCompressStream;
    const QzDecompressStreamFn qzDecompressStream;
    const QzEndStreamFn qzEndStream;
    const QzGetSoftwareComponentVersionListFn qzGetSoftwareComponentVersionList;
    const QzGetSoftwareComponentCountFn qzGetSoftwareComponentCount;
    const QzSetupSessionDeflateFn qzSetupSessionDeflate;
    const QzSetupSessionLZ4Fn qzSetupSessionLZ4;
    const QzSetupSessionLZ4SFn qzSetupSessionLZ4S;
    const QzSetDefaultsDeflateFn qzSetDefaultsDeflate;
    const QzSetDefaultsLZ4Fn qzSetDefaultsLZ4;
    const QzSetDefaultsLZ4SFn qzSetDefaultsLZ4S;
    const QzGetDefaultsDeflateFn qzGetDefaultsDeflate;
    const QzGetDefaultsLZ4Fn qzGetDefaultsLZ4;
    const QzGetDefaultsLZ4SFn qzGetDefaultsLZ4S;
    const QzCompressExtFn qzCompressExt;
    const QzDecompressExtFn qzDecompressExt;
    const QzCompressCrc64Fn qzCompressCrc64;
    const QzCompressCrc64ExtFn qzCompressCrc64Ext;
    const QzDecompressCrcFn qzDecompressCrc;
    const QzDecompressCrcExtFn qzDecompressCrcExt;
    const QzDecompressCrc64Fn qzDecompressCrc64;
    const QzDecompressCrc64ExtFn qzDecompressCrc64Ext;
    const QzGetSessionCrc64Config qzGetSessionCrc64Config;
    const QzSetSessionCrc64Config qzSetSessionCrc64Config;
    const QzAllocateMetadataFn qzAllocateMetadata;
    const QzFreeMetadataFn qzFreeMetadata;
    const QzMetadataBlockReadFn qzMetadataBlockRead;
    const QzMetadataBlockWriteFn qzMetadataBlockWrite;
    const QzCompressWithMetadataExtFn qzCompressWithMetadataExt;
    const QzDecompressWithMetadataExtFn qzDecompressWithMetadataExt;
    const QzGetSessionCrc32Config qzGetSessionCrc32Config;
    const QzSetSessionCrc32Config qzSetSessionCrc32Config;
    const QzMetadataBlockGetCrc64Fn qzMetadataBlockGetCrc64;
    const QzMetadataBlockGetCrc32Fn qzMetadataBlockGetCrc32;
} QZ_FUNCTION_TABLE, *PQZ_FUNCTION_TABLE;

typedef struct _QZ_IOCTL_OUTPUT_PARAMS
{
    PVOID pDispatchTable;
} QZ_IOCTL_OUTPUT_PARAMS, *PQZ_IOCTL_OUTPUT_PARAMS;

/**
 *****************************************************************************
 * This structure is passed as an output to IOCTL_GET_QZ_INTERFACE_INFO
 * and as an input to IOCTL_GET_QZ_INTERFACE_EXT.
 */
typedef struct _QZ_IOCTL_INTERFACE_INFO
{
    uint32_t version;
    /* Driver version. */
    uint32_t qzInterfaceSize;
    /* Size of the intreface table.
     */
} QZ_IOCTL_INTERFACE_INFO, *PQZ_IOCTL_INTERFACE_INFO;

#define IOCTL_GET_QZ_INTERFACE \
    CTL_CODE(FILE_DEVICE_UNKNOWN, 0x906, METHOD_BUFFERED, FILE_ANY_ACCESS)
/**
 ******************************************************************************
 * IOCTL to get the interface table size for a given version.
 * version is to be passed as an input and @QZ_IOCTL_INTERFACE_INFO
 * structure is to be passed as an output buffer.
 * If version is greater than QATZIP_KERNEL_API_VERSION,
 * then STATUS_NOT_SUPPORTED is returned and driver populates
 * @QZ_IOCTL_INTERFACE_INFO.version with QATZIP_KERNEL_API_VERSION
 * and @QZ_IOCTL_INTERFACE_INFO.qzInterfaceSize with QATZIP_KERNEL_API_VERSION
 * interface table size.
 * If version is valid, then @QZ_IOCTL_INTERFACE_INFO.version is filled
 * with user input version and @QZ_IOCTL_INTERFACE_INFO.qzInterfaceSize
 * is filled with corresponding interface table size.
 */
#define IOCTL_GET_QZ_INTERFACE_INFO \
    CTL_CODE(FILE_DEVICE_UNKNOWN, 0x90B, METHOD_NEITHER, FILE_ANY_ACCESS)
/**
 ******************************************************************************
 * IOCTL to retrieve the interface table entries for a given version
 * and a table entries memory of size qzInterfaceSize.
 * @QZ_IOCTL_INTERFACE_INFO is to be passed as an input and
 * a buffer of @QZ_IOCTL_INTERFACE_INFO.qzInterfaceSize is to be
 * passed as output buffer.
 * If @QZ_IOCTL_INTERFACE_INFO.version is greater than
 * QATZIP_KERNEL_API_VERSION, then STATUS_NOT_SUPPORTED is returned.
 * If @QZ_IOCTL_INTERFACE_INFO.qzInterfaceSize doesn't match
 * with the interface table size in driver, then STATUS_INVALID_BUFFER_SIZE
 * is returned. If both version and qzInterfaceSize are valid, then
 * the table entries would be copied to the output buffer.
 */
#define IOCTL_GET_QZ_INTERFACE_EXT \
    CTL_CODE(FILE_DEVICE_UNKNOWN, 0x90C, METHOD_NEITHER, FILE_ANY_ACCESS)

/* The current maximum version supported by driver. */
#define QATZIP_KERNEL_API_VERSION 1

/* Version that supports CRC. */
#define MIN_QATZIP_KERNEL_API_VERSION_CRC_SUPPORT 1

/* The default size(in bytes) of interface table before
 * MIN_QATZIP_KERNEL_API_VERSION_CRC_SUPPORT is added in driver.
 */
#define QATZIP_KERNEL_API_VERSION_0_INTERFACE_SIZE 352

#endif // QATZIP_KM_H_
